# Introduction

## Jumping Right In : *What Pawpaw can do for you*

Say you have some text:

```python
>>> s = 'nine 9 ten 10 eleven 11 TWELVE 12 thirteen 13'
```

and you wish to perform an nlp-like segmentation on it to identify phrases, words, numbers, and so forth.  One approach is to use a regular expression:

```python
>>> import regex 
>>> re = regex.compile(r'(?:(?P<phrase>(?P<word>(?P<char>\w)+) (?P<number>(?P<digit>\d)+))\s*)+')
>>> m = re.fullmatch(s)
```

You can then feed the match object to **pawpaw** as follows:

```python
>>> import pawpaw
>>> root = pawpaw.Ito.from_match(m)[0]  # take first
```

With just this one line of code, Pawpaw creates a fully hierarchical, tree[^tree_graph] of phrases, words, chars, numbers, and digits.  This tree can be traversed, and even searched using *plumule*, a powerful XPATH-like structured query language:

```python
>>> print(*root.find_all('**[d:digit]'), sep=', ')  # print all digits
9, 1, 0, 1, 1, 1, 2, 1, 3
>>> print(*root.find_all('**[d:number]{</*[s:i]}'), sep=',')  # print numbers having letter 'i' in their names
9, 13
```

## Dive a Bit Deeper: *Introducing* ``Ito``

```python
>>> from pawpaw import Ito
>>> s = 'x Hello, world! x'
>>> i = Ito(s, 2, -2)
>>> print(i)
Hello, world!
```

The class ``Ito``, short for *In Text Object*, is a high performance, indexed substring. Creating
an ``Ito`` does *not* create a new string.  Rather, an ``Ito`` instance maintains a reference to its
basis string, along with start and stop indexes:

```python
>>> i
Ito(span=(2, 15), desc='', substr='Hello, world!')
```

Because ``Ito`` only maintains a reference to its basis string, its memory requirements are low:

```python
>>> import sys
>>> sys.getsizeof(s), sys.getsizeof(i)
(66, 48)
>> s2 = s * 4096  # a big str
>>> sys.getsizeof(s2), sys.getsizeof(Ito(s2))
(69681, 48)
```

### ``str``-like Behavior

The class ``Ito`` behaves like a ``str`` whenever possible.  For example, you can print and Ito, or use it with a f-style string:

#### Formatted Output

```python
>>> print(i)
Hello, world!
>>> f'{i}'
'Hello, world!'
```

``Ito`` also supports custom format strings.  You can quickly develop custom outputs by combining and formatting ``Ito``
attributes in various ways, and even perform truncations to facilitate visualizations:

```python
>>> j = Ito('Here is\n a super\nduper, very long string!')
>>> f'{j:%span% : %substr!r:21…\'}'  # Show span and substr
"(0, 41): 'Here is\\n a super\\…'"
```

#### ``len``

You can access the length of an `Ito` using the keyword `len`:

```python
>>> len(i)
13
```

#### Indexing and Slicing

As with strings, you can access character data via ``int`` or ``slice`` indexers[^slice_step].  Note that indexing an ``Ito`` is always *relative to itself*, not its basis string:

```python
  >>> str(i[0])    # int accessor
H
>>> str(i[-1])   # int accessor
!
>>> str(i[:5])   # slice accessor
Hello
>>> str(i[-6:])  # slice accessor
world!
```
  
Just as index access on a ``str`` returns another ``str``, index access on an ``Ito`` returns another ``Ito``:

```python
>>> i[-6:]
Ito(span=(9, 15), desc='', substr='world!')
```

Relative index access allows for clear, easy to use code:

```python
>>> j = i[-6:]   # new Ito based on the last 6 characters of the previous Ito
>>> str(j[0:3])  # print first three characters of the new Ito
```

An ``Ito`` is iterable in the same way a Python ``str`` is:

```python
>>> for idx, ito in enumerate(i[:5]):
...     print(idx, ito):
...
0: H
1: e
2: l
3: l
4: o
```

### Instantiation

``Ito`` constructors allow either a ``str`` or an ``Ito`` as the basis parameter, ``src``.  The subsequent ``start`` and ``stop`` parameters are always *relative to ``src``*, with the original source string carried through and all indexing bookkeeping handled automatically:

```python
>>> s = '_abc_'
>>> i1 = Ito(s, 1, -1)   # str basis; skip first & last chars of str s
>>> i2 = Ito(i1, 1, -1)  # Ito basis; skip first and last chars of Ito i1
>>> str(i1), str(i2)
('abc', 'b')
```

Multiple static constructor methods are available to create and ``Ito`` from things such as:

* regex.Match
* spans or gaps
* ordinary substrings

### Descriptor
``Ito`` features a ``.desc`` property that can be used for tagging.  Values for ``.desc`` can be specified as the final constructor property, or post-instantiation via a property of the same name::

 ```python
>>> i = Ito('abc', desc='something useful')
 >>> i.desc
 'something useful'
 >>> i.desc = 'something different'
 >>> i.desc
 'something different'
```

 The ``.desc`` property facilitates search and retrieval of ``Ito`` collections.
 
### Tree Data Store

``Ito`` supports nested storage of child ``Ito`` instances via the ``.children`` property.  Nodes can be added via the ``.add`` method in arbitrary order.  They are always stored in sorted order:

```python
>>> s = 'ABC 123'
>>> root = Ito(s)
>>> root.children.add(root[-3:], root[:3])
>>> [str(c) for c in root.children]
['ABC', '123']
```

The ``.children`` collection is not an arbitrary data store.  Nodes added to it must meet the following criteria:

* An instance (or sublass) of ``Ito``
* Have the same reference ``str`` as the parent
* Not overlap any existing child nodes
* Be non-empty

Child nodes have their own ``.children`` collections, and so on.  In other words, ``Ito`` and its descendants form are hierarchical and from a tree graph.  This quality  is used for many key features of Pawpaw, such as:

* Storage and organization of substring data
* Hierarchical segmentation processing
* Traversal of substring collections
* Structured search and query

## Arborform

Pawpaw makes it easy to create trees.  As seen in the introduction above, a fully formed tree of arbitrary depth can be formed using a single regular expression.  Creating large regular expressions, however, can sometimes be unwieldy.  Pawpaw's ``.arborform`` namespace contains wide variety of tools to help you quicly build trees using pipelining.  For example, the same tree shown in the Introduction can be created in chained, piecemeal fashion as follows::

```python
>>> import regex
>>> from pawpaw import Ito, arborform, visualization
>>>
>>> s = 'nine 9 ten 10 eleven 11 TWELVE 12 thirteen 13'
>>>
>>> root = Ito(s, desc='root')
>>>
>>> phrases = arborform.Split(regex.compile(r'(?<=\d)'), desc='phrase')
>>>
>>> wrds_nums = arborform.Extract(regex.compile(r'(?P<word>[a-z]+) (?P<number>\d+)'))
>>> con = arborform.Connectors.Children.Add(wrds_nums)
>>> phrases.connections.append(con)
>>>
>>> chrs_digs = arborform.Extract(regex.compile(r'(?P<char>[a-z])+|(?P<digit>\d)+'))
>>> con = arborform.Connectors.Children.Add(chrs_digs)
>>> wrds_nums.connections.append(con)
>>>
>>> root.children.add(*phrases(root))
>>>
>>> tree_vis = visualization.pepo.Tree()
>>> print(tree_vis.dumps(root))
(0, 45) 'root' : 'nine 9 ten 10 eleven…ELVE 12 thirteen 13'
├──(0, 6) 'Phrase' : 'nine 9'
│  ├──(0, 4) 'word' : 'nine'
│  │  ├──(0, 1) 'char' : 'n'
│  │  ├──(1, 2) 'char' : 'i'
│  │  ├──(2, 3) 'char' : 'n'
│  │  └──(3, 4) 'char' : 'e'
│  └──(5, 6) 'number' : '9'
│     └──(5, 6) 'digit' : '9'
├──(6, 12) 'Phrase' : ' ten 1'
│  ├──(7, 10) 'word' : 'ten'
│  │  ├──(7, 8) 'char' : 't'
│  │  ├──(8, 9) 'char' : 'e'
│  │  └──(9, 10) 'char' : 'n'
│  └──(11, 12) 'number' : '1'
│     └──(11, 12) 'digit' : '1'
├──(12, 13) 'Phrase' : '0'
├──(13, 22) 'Phrase' : ' eleven 1'
│  ├──(14, 20) 'word' : 'eleven'
│  │  ├──(14, 15) 'char' : 'e'
│  │  ├──(15, 16) 'char' : 'l'
│  │  ├──(16, 17) 'char' : 'e'
│  │  ├──(17, 18) 'char' : 'v'
│  │  ├──(18, 19) 'char' : 'e'
│  │  └──(19, 20) 'char' : 'n'
│  └──(21, 22) 'number' : '1'
│     └──(21, 22) 'digit' : '1'
├──(22, 23) 'Phrase' : '1'
├──(23, 32) 'Phrase' : ' TWELVE 1'
├──(32, 33) 'Phrase' : '2'
├──(33, 44) 'Phrase' : ' thirteen 1'
│  ├──(34, 42) 'word' : 'thirteen'
│  │  ├──(34, 35) 'char' : 't'
│  │  ├──(35, 36) 'char' : 'h'
│  │  ├──(36, 37) 'char' : 'i'
│  │  ├──(37, 38) 'char' : 'r'
│  │  ├──(38, 39) 'char' : 't'
│  │  ├──(39, 40) 'char' : 'e'
│  │  ├──(40, 41) 'char' : 'e'
│  │  └──(41, 42) 'char' : 'n'
│  └──(43, 44) 'number' : '1'
│     └──(43, 44) 'digit' : '1'
└──(44, 45) 'Phrase' : '3'
```

[^tree_graph]: A tree is an undirected graph in which any two vertices are connected by exactly one path.  An ``Ito`` is technically a *rooted tree*, where the root is the ``Ito`` and the branches and leaves its descendants.

[^slice_step]: Slices with step values other than 1 (or ``None``) are not supported because the resulting sequences of characters will typically *not* be valid substrings.  For example, ``'abc[::-1]`` results in the reversed string ``cba``, which doesn't align with the starting ``str``.
